import os
from flask import Flask, request, render_template, jsonify, flash, redirect, url_for, g
import mysql.connector
from dotenv import load_dotenv

# --- PRODUCTION-READY .env LOADING ---
project_folder = os.path.dirname(os.path.abspath(__file__))
dotenv_path = os.path.join(project_folder, '.env')
load_dotenv(dotenv_path=dotenv_path)

# --- FLASK APP INITIALIZATION ---
app = Flask(__name__)
application = app  # for cPanel WSGI

# --- CONFIGURATION ---
app.config['MYSQL_HOST'] = os.getenv('MYSQL_HOST')
app.config['MYSQL_USER'] = os.getenv('MYSQL_USER')
app.config['MYSQL_PASSWORD'] = os.getenv('MYSQL_PASSWORD')
app.config['MYSQL_DB'] = os.getenv('MYSQL_DB')
app.secret_key = os.getenv('SECRET_KEY')

# --- IMPROVED DATABASE CONNECTION MANAGEMENT ---
def get_db():
    """Opens a new database connection if there is none yet for the current application context."""
    if 'db' not in g:
        g.db = mysql.connector.connect(
            host=app.config['MYSQL_HOST'],
            user=app.config['MYSQL_USER'],
            password=app.config['MYSQL_PASSWORD'],
            database=app.config['MYSQL_DB']
        )
    return g.db

@app.teardown_appcontext
def close_db(e=None):
    """Closes the database again at the end of the request."""
    db = g.pop('db', None)
    if db is not None:
        db.close()

# --- BLUEPRINTS ---
from about import about_bp
from information import information_bp
from wewithyou import wewithyou_bp
from tour import tour_bp
from service import service_bp
from gallery import gallery_bp

app.register_blueprint(about_bp)
app.register_blueprint(information_bp)
app.register_blueprint(wewithyou_bp)
app.register_blueprint(tour_bp)
app.register_blueprint(service_bp)
app.register_blueprint(gallery_bp)

# --- GLOBAL CONTEXT PROCESSORS ---
@app.context_processor
def inject_nav_tours():
    try:
        conn = get_db()
        cur = conn.cursor(dictionary=True)
        cur.execute("SELECT title, slug FROM tours")
        nav_tours = cur.fetchall()
        cur.close()
        return dict(nav_tours=nav_tours)
    except Exception as e:
        print(f"[NAV TOURS] Error: {e}")
        return dict(nav_tours=[])

# --- CORE ROUTES ---
@app.route('/')
def home():
    tours = []
    news_items = []
    try:
        conn = get_db()
        cur = conn.cursor(dictionary=True)

        cur.execute("""
            SELECT id, title, category, image, slug, content 
            FROM tours 
            WHERE visibility = 1 AND is_the_best_package = 1
            ORDER BY created_at DESC 
            LIMIT 4
        """)
        tours = cur.fetchall()

        cur.execute("""
            SELECT id, title, description, source, jdate, created_at 
            FROM news_events 
            WHERE visibility = 1 
            ORDER BY created_at DESC 
            LIMIT 5
        """)
        news_items = cur.fetchall()

        cur.close()
    except Exception as e:
        print(f"Error fetching data for homepage: {e}")
    return render_template('pages/home.html', tours=tours, news_items=news_items)

@app.route('/about')
def about():
    return render_template('pages/about.html')

@app.route('/privacy-policy')
def privacy_policy():
    return render_template('pages/privacy-policy.html')

@app.route('/reviews')
def reviews():
    return render_template('pages/reviews.html')

# ---- Contact Us -----
@app.route('/contact-us', methods=['GET', 'POST'])
def contact_us():
    if request.method == 'POST':
        try:
            fname = request.form['fname']
            lname = request.form['lname']
            email = request.form['email']
            mobile = request.form['mobile']
            message = request.form['message']

            conn = get_db()
            cur = conn.cursor()
            cur.execute("INSERT INTO contacts(fname, lname, email, mobile, message) VALUES(%s, %s, %s, %s, %s)",
                        (fname, lname, email, mobile, message))
            conn.commit()
            cur.close()

            flash('Thank you for your message! We will get back to you shortly.', 'success')
        except Exception as e:
            print(f"Database Error: {e}")
            flash('An error occurred while submitting your message. Please try again.', 'danger')
        return redirect(url_for('contact_us'))
    
    return render_template('pages/contactus.html')

# ---- Book Now -----
@app.route('/booking', methods=['GET', 'POST'])
def booking():
    if request.method == 'POST':
        try:
            fname = request.form['fname']
            lname = request.form['lname']
            email = request.form['email']
            mobile = request.form['mobile']
            persons = request.form['members']
            places = request.form['places']
            arrival_date = request.form['arrival_date']
            message = request.form['message']

            sql_query = """
                INSERT INTO bookings(fname, lname, email, mobile, message, persons, places, arrival_date) 
                VALUES(%s, %s, %s, %s, %s, %s, %s, %s)
            """
            values = (fname, lname, email, mobile, message, persons, places, arrival_date)

            conn = get_db()
            cur = conn.cursor()
            cur.execute(sql_query, values)
            conn.commit()
            cur.close()

            flash('Your Booking Request is Submitted Successfully! We will get back to you shortly.', 'success')
        except Exception as e:
            print(f"Database Error: {e}")
            flash('An error occurred while submitting your booking. Please try again.', 'danger')
        return redirect(url_for('booking'))
    
    return render_template('pages/booking_page.html')


# The following block is for local development only.
# It MUST be removed or commented out for production deployment.
# if __name__ == '__main__':
#     app.run(debug=True)